﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using VeteransAffairs.Registries.BusinessAHOBPR;
using VeteransAffairs.Registries.BusinessAHOBPR.VeteranStatusService;


namespace VeteransAffairs.Registries.BusinessManagerAHOBPR.Emis
{
    public class VeteranStatusImporter : AHOBPRBaseBO, IRegistrantUpdater
    {

        eMISVeteranStatusServicePortTypesClient _client;
        InputHeaderInfo _inputHeaderInfo;
        public VeteranStatusImporter()
        {
            _client = new eMISVeteranStatusServicePortTypesClient();
            _inputHeaderInfo = new InputHeaderInfo();
        }

        #region Properties

        public string Name { get { return "VeteranStatus"; } }

        #endregion


        public bool Update(string edipi, int registrantId)
        {
            var successfullyProcessed = true;
            try
            {
                //Get veteranstatus info from emis using edipi
                VeteranStatus data = GetVeteranStatus(edipi);
                if (data != null)
                {
                    successfullyProcessed = ProcessVeteranStatus(data);
                }
                else
                {
                    var errormessage = "VeteranStatusImporter.Update(): null VeteranStatus object returned when calling VeteranStatusImporter.GetVeteranStatus() for registrant with registrant_id: " + registrantId;
                    Trace.WriteLine(errormessage);
                    AHOBPRLogger.LogErrorMessage("eMIS Error", this.GetType().Name + "." + System.Reflection.MethodBase.GetCurrentMethod().Name, errormessage);
                    successfullyProcessed = false;
                }
                //update registrant object with veteranstatus information
            }
            catch (Exception ex)
            {
                successfullyProcessed = false;
            }

            return successfullyProcessed;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="data"></param>
        /// <returns></returns>
        bool ProcessVeteranStatus(VeteranStatus data)
        {
            int returnStatus = 0;
            //if not null, get a registrant from the DB
            if (data != null)
            {
                var manager = new AHOBPRRegistrantManager();
                var registrant = manager.GetRegistrantById(data.edipi, "");
                //Check if Registrant is null, otherwise this is all pointless. 
                if (registrant != null)
                {
                     UpdateRegistrantWithVeteranStatusInfo(data, registrant);
                     returnStatus = manager.UpdateRegistrant(registrant);
                }
                else //The registrant object REALLY SHOULDN'T be null, so add to the log and call shennanigans.
                {
                    var errormessage = "Database call couldn't find registrant by EDIPI: " + data.edipi;
                    Trace.WriteLine("VeteranStatusImporter.ProcessVeteranStatus: " + errormessage);
                    AHOBPRLogger.LogErrorMessage("DB Error", this.GetType().Name + "." + System.Reflection.MethodBase.GetCurrentMethod().Name, errormessage);
                    returnStatus = -1;
                }
            }
            //A returnStatus of -1 means the attempt to save to the DB failed. Anything else is the ammount and attempts to save to the DB from 1 through 20
            return (returnStatus != -1);
        }

     
        private VeteranStatus GetVeteranStatus(string edipi)
        {
            
            var inputEdipiIcn = (inputEdiPiOrIcn)InputEdipiOrIcnCreator.Create(edipi);
            eMISveteranStatusResponseType response = GetVeteranStatusResponse(inputEdipiIcn, _inputHeaderInfo);
            if (response.ESSError != null)
            {
                Trace.WriteLine("ESSError retreiving response from eMIS. VeteranStatusImporter.GetVeteranStatus: " + response.ESSError.text);
                AHOBPRLogger.LogErrorMessage("ESSERROR calling GetVeteranStatus", this.GetType().Name + "." + System.Reflection.MethodBase.GetCurrentMethod().Name, response.ESSError.text);
            }
            return response.veteranStatus;
        }

        private eMISveteranStatusResponseType GetVeteranStatusResponse(inputEdiPiOrIcn edipi, InputHeaderInfo inputHeaderInfo)
        {
            eMISveteranStatusResponseType response = null;
            try
            {
                response = _client.getVeteranStatus(ref inputHeaderInfo, edipi);
            }
            catch (Exception ex)
            {
                Trace.WriteLine("Exception: VeteranStatusImporter.GetVeteranStatusResponse: " + ex.Message + "\n" + ex.StackTrace);
                AHOBPRLogger.LogErrorMessage("Exception", this.GetType().Name + "." + System.Reflection.MethodBase.GetCurrentMethod().Name, ex.StackTrace);
            }
            return response;
        }
        
        /// <summary>
        /// 
        /// </summary>
        /// <param name="data"></param>
        /// <param name="registrant"></param>
        /// <returns></returns>
        void UpdateRegistrantWithVeteranStatusInfo(VeteranStatus data, REGISTRANT registrant)
        {

            Trace.WriteIf(data.socialSecurityNumber != registrant.Snum, $"Registrant: {registrant.REGISTRANT_ID}'s social security number does not match with eMIS's social security number");
            Trace.WriteIf(data.personFirstName.ToLower() != registrant.FIRST_NAME.ToLower(), $"Registrant: {registrant.REGISTRANT_ID}'s first name does not match eMIS's entry.");
            Trace.WriteIf(data.personLastName.ToLower() != registrant.LAST_NAME.ToLower(), $"Registrant: {registrant.REGISTRANT_ID}'s last name does not match eMIS's entry.");
            Trace.WriteIf(data.personBirthDate != registrant.BIRTH_DATE, $"Registrant: {registrant.REGISTRANT_ID}'s birth date does not match eMIS's entry.");

            UpdateMaritalStatus(data, registrant);
            UpdateEducationLevel(data, registrant);
            UpdateRace(data, registrant);
            UpdateEthnicAffinityCode(data, registrant);
            UpdateDeathIndicator(data, registrant);
            UpdateDeathDate(data, registrant);
            Update911DeploymentIndicators(data, registrant);
        }

              #region Registrant fields to update 

        private void UpdateDeathDate(VeteranStatus data, REGISTRANT registrant)
        {
            if (data.personDeathDate != DateTime.Parse("0001-01-01") && !data.personDeathDate.Equals(registrant.DEATH_DATE))
            {
                registrant.DEATH_DATE = data.personDeathDate;
            }
        }

        private void UpdateDeathIndicator(VeteranStatus data, REGISTRANT registrant)
        {
            registrant.DECEASED_FLAG = TranslateDeathIndicatorCode(data.personDeathIndicatorCode);
        }



        private void UpdateEthnicAffinityCode(VeteranStatus data, REGISTRANT registrant)
        {
            if (ShouldUpdateTheField(data.ethnicAffinityCode, registrant.ETHNICITY)) { }
        }

        private static void UpdateMaritalStatus(VeteranStatus data, REGISTRANT registrant)
        {
            if (ShouldUpdateTheField(data.maritalStatusCode, registrant.MARITAL_STATUS))
            {
                registrant.MARITAL_STATUS = data.maritalStatusCode;
            }
        }


        private static void UpdateEducationLevel(VeteranStatus data, REGISTRANT registrant)
        {
            if (ShouldUpdateTheField(data.educationLevelCode, registrant.EDUCATION_LEVEL))
            {
                registrant.EDUCATION_LEVEL = data.educationLevelCode;
            }
        }

        private static void UpdateRace(VeteranStatus data, REGISTRANT registrant)
        {
            if (ShouldUpdateTheField(data.raceCode, registrant.RACE))
            {
                registrant.RACE = data.raceCode;
            }
        }

        static void Update911DeploymentIndicators(VeteranStatus data, REGISTRANT registrant)
        {
            if (ShouldUpdateTheField(data.pre911DeploymentIndicator, registrant.PRE_911_DEPLOYMENT))
            {
                registrant.PRE_911_DEPLOYMENT = data.pre911DeploymentIndicator;
            }
            if (ShouldUpdateTheField(data.post911DeploymentIndicator, registrant.POST_911_DEPLOYMENT))
            {
                registrant.POST_911_DEPLOYMENT = data.post911DeploymentIndicator;
            }
        }
        #endregion

        #region helper methods
        bool TranslateDeathIndicatorCode(string code)
        {
            return code.ToLower().Equals("y");
        }
        static bool ShouldUpdateTheField(string dataField, string registrantField)
        {
            var result = false;
            //Don't update a possibly populated registrant field if the data field is empty
            if (!String.IsNullOrEmpty(dataField))
            {
                result = !dataField.Equals(registrantField, StringComparison.OrdinalIgnoreCase);
            }
            return result;
        }
        #endregion   
    }
}
